import 'dart:async';
import 'dart:math';

import 'package:cached_network_image/cached_network_image.dart';
import 'package:flutter/material.dart';

import 'package:photogram/import/core.dart';
import 'package:photogram/import/bloc.dart';
import 'package:photogram/import/data.dart';
import 'package:photogram/import/interface.dart';

import 'package:photogram/theme/photogram/include/pg_utils.dart';

class PgProfileUserTaggedInPostsGridView extends ProfileUserPostsPage {
  final int userId;
  final String instanceStateId;

  // final StreamController<bool> streamController;

  const PgProfileUserTaggedInPostsGridView({
    Key? key,
    required this.userId,
    required this.instanceStateId,
    // required this.streamController,
  }) : super(key: key);

  @override
  _PgProfilePostsPageGridState createState() => _PgProfilePostsPageGridState();
}

class _PgProfilePostsPageGridState extends State<PgProfileUserTaggedInPostsGridView>
    with
        AutomaticKeepAliveClientMixin<PgProfileUserTaggedInPostsGridView>,
        AppActiveContentInfiniteMixin,
        AppUtilsMixin {
  late UserModel _userModel;

  var _previousInstanceStateId = '';

  @override
  bool get wantKeepAlive {
    var doReload = _previousInstanceStateId != widget.instanceStateId;

    if (doReload) {
      _previousInstanceStateId = widget.instanceStateId;

      utilMixinSetState(() {
        contentMixinReloadPage();
      });

      return false;
    }

    return true;
  }

  final _postUserTagIds = <int>[];

  @override
  void onLoadEvent() {
    _userModel = activeContent.watch<UserModel>(widget.userId) ?? UserModel.none();

    if (_postUserTagIds.isEmpty) {
      _loadPosts(latest: true);
    }
  }

  @override
  onReloadBeforeEvent() {
    _postUserTagIds.clear();
    return true;
  }

  @override
  onReloadAfterEvent() {
    _loadPosts(latest: true);
  }

  @override
  Widget build(BuildContext context) {
    super.build(context);

    if (_userModel.isNotModel) {
      return AppLogger.fail('${_userModel.runtimeType}(${widget.userId})');
    }

    if (_postUserTagIds.isEmpty && !isLoadingCallInStack) {
      return _buildInfoSection();
    }

    return Column(
      children: [
        if (isLoadingLatest) Expanded(child: PgUtils.gridImagePreloaderFull(3)),
        Expanded(
          child: GridView.builder(
            itemCount: isLoadingBottom ? _postUserTagIds.length + 12 : _postUserTagIds.length,
            itemBuilder: _buildPost,
            gridDelegate: const SliverGridDelegateWithFixedCrossAxisCount(
              crossAxisCount: 3,
              crossAxisSpacing: 2,
              mainAxisSpacing: 2,
            ),
          ),
        ),
      ],
    );
  }

  Widget _buildInfoSection() {
    return ListView(
      shrinkWrap: true,
      padding: const EdgeInsets.all(32),
      children: [
        Center(
          child: ConstrainedBox(
            constraints: const BoxConstraints(maxWidth: 250),
            child: Column(
              children: [
                Text(
                  _userModel.isLoggedIn
                      ? AppLocalizations.of(context)!.photosOfYou
                      : AppLocalizations.of(context)!.taggedInPosts,
                  style: ThemeBloc.textInterface.boldBlackH2TextStyle(),
                  textAlign: TextAlign.center,
                ),
                PgUtils.sizedBoxH(10),
                Text(
                  _userModel.isLoggedIn
                      ? AppLocalizations.of(context)!.whenPeopleTagYouInPhotosTheyIllAppearHere
                      : AppLocalizations.of(context)!.userIsntTaggedInAnyPost,
                  style: ThemeBloc.textInterface.normalGreyH5TextStyle(),
                  textAlign: TextAlign.center,
                ),
              ],
            ),
          ),
        ),
      ],
    );
  }

  Widget _buildPost(BuildContext context, int index) {
    // agressive prefetching posts
    if (_postUserTagIds.length - 3 < index) {
      _loadPosts(waitForFrame: true);
    }

    if (_postUserTagIds.length > index) {
      var postUserTagModel = activeContent.watch<PostUserTagModel>(_postUserTagIds[index]) ?? PostUserTagModel.none();
      if (postUserTagModel.isNotModel) {
        return AppLogger.fail('${postUserTagModel.runtimeType}');
      }

      var postIdToBuild = AppUtils.intVal(postUserTagModel.taggedInPostId);

      var postModel = activeContent.watch<PostModel>(postIdToBuild) ?? PostModel.none();
      if (postModel.isNotModel) {
        return AppLogger.fail('${postModel.runtimeType}($postIdToBuild)');
      }
      if (!postModel.isVisible) {
        return PgUtils.gridImageStationary();
      }

      return GestureDetector(
        onTap: () {
          AppNavigation.push(
            context,
            MaterialPageRoute(
              builder: (context) {
                return ThemeBloc.pageInterface.profileUserTaggedInPostsPage(
                  userId: widget.userId,
                  postUserTagIds: _postUserTagIds,
                  scrollToPostUserTagIndex: index,
                );
              },
            ),
            utilMixinSetState,
          );
        },
        child: Stack(
          children: [
            AspectRatio(
              aspectRatio: 1,
              child: CachedNetworkImage(
                placeholder: (_, __) => Container(color: Colors.grey[300]),
                imageUrl: postModel.getFirstImageUrlFromPostContent,
                fit: BoxFit.cover,
              ),
            ),
            if (postModel.displayContent.length > 1) PgUtils.postContentIcon()
          ],
        ),
      );
    }

    // add prelaoder
    return PgUtils.gridImagePreloader();
  }

  Future<void> _loadPosts({bool latest = false, bool waitForFrame = false}) async {
    contentMixinLoadContent(
      latest: latest,
      waitForFrame: waitForFrame,
      responseHandler: handleResponse,
      latestEndpoint: REQ_TYPE_POST_USER_TAG_LOAD_LATEST,
      bottomEndpoint: REQ_TYPE_POST_USER_TAG_LOAD_BOTTOM,
      requestDataGenerator: () => {
        PostUserTagTable.tableName: {PostUserTagTable.taggedUserId: _userModel.id},
        RequestTable.offset: {
          PostUserTagTable.tableName: {PostUserTagTable.id: latest ? latestContentId : bottomContentId},
        },
      },
    );
  }

  handleResponse({
    bool latest = false,
    required ResponseModel responseModel,
  }) {
    activeContent.handleResponse(responseModel);

    setState(() {
      activeContent.pagedModels<PostUserTagModel>().forEach((postUserTagId, postUserTagModel) {
        if (!_postUserTagIds.contains(postUserTagId)) {
          _postUserTagIds.add(postUserTagId);
        }
      });

      contentMixinUpdateData(
        setLatestContentId: _postUserTagIds.isEmpty ? 0 : _postUserTagIds.reduce(max),
        setBottomContentId: _postUserTagIds.isEmpty ? 0 : _postUserTagIds.reduce(min),
      );
    });
  }
}
